<?php

class ITSEC_Malware_Scan_Results_Template {
	public static function get_html( $results, $show_error_details = false ) {

		if ( ( $html = apply_filters( 'itsec_pre_malware_scanner_get_html', null, $results, $show_error_details ) ) !== null ) {
			return $html;
		}

		if ( is_wp_error( $results ) && ( $edata = $results->get_error_data() ) && ! empty( $edata['itsec_site'] ) ) {
			$site_url = $edata['itsec_site']['url'];
		} elseif ( is_array( $results ) && ! empty( $results['itsec_site'] ) ) {
			$site_url = $results['itsec_site']['url'];
		} else {
			$site_url = '';
		}

		$html = "<div class='itsec-malware-scan-results'>\n";

		if ( $site_url ) {
			$html .= '<h4>' . sprintf( esc_html__( 'Site: %s', 'better-wp-security' ), $site_url ) . '</h4>';
		}

		if ( is_wp_error( $results ) ) {
			$html .= self::get_wp_error_details( $results, $show_error_details );
		} else {
			$html .= self::get_system_error_details( $results );
			$html .= self::get_malware_details( $results );
			$html .= self::get_blacklist_details( $results );
		}

		$html .= "</div>\n";

		return $html;
	}

	protected static function get_wp_error_details( $results, $show_error_details ) {
		$status = 'error';
		$description = __( 'The scan failed to properly scan the site.', 'better-wp-security' );

		$details = '<p>' . sprintf( __( 'Error Message: %s', 'better-wp-security' ), $results->get_error_message() ) . "</p>\n";
		$details .= '<p>' . sprintf( __( 'Error Code: <code>%s</code>', 'better-wp-security' ), esc_html( $results->get_error_code() ) ) . "</p>\n";

		if ( $show_error_details ) {
			$data = $results->get_error_data();

			if ( ! empty( $data ) ) {
				$details .= '<p>' . __( 'If you contact support about this error, please provide the following debug details:', 'better-wp-security' ) . "</p>\n";

				$details .= '<pre>' . esc_html( print_r( array( 'code' => $results->get_error_code(), 'data' => $data ), true ) ) . "</pre>\n";
			}
		}

		return self::get_wrapped_section( 'wp-error', $status, $description, $details );
	}

	protected static function get_blacklist_details( $results ) {
		$status = 'clean';
		$description = __( 'Blacklist', 'better-wp-security' );
		$details = false;

		$list = '';

		if ( ! empty( $results['BLACKLIST']['WARN'] ) ) {
			$status = 'warn';

			foreach ( (array) $results['BLACKLIST']['WARN'] as $entry ) {
				$list .= sprintf( '<li class="itsec-malware-scan-warn"><span><a href="%1$s">%2$s</a><span></li>', $entry[1], $entry[0] ) . "\n";
			}
		}

		if ( ! empty( $results['BLACKLIST']['INFO'] ) ) {
			foreach ( (array) $results['BLACKLIST']['INFO'] as $entry ) {
				$list .= sprintf( '<li class="itsec-malware-scan-clean"><span><a href="%1$s">%2$s</a></span></li>', $entry[1], $entry[0] ) . "\n";
			}
		}

		if ( ! empty( $list ) ) {
			$details = "<ul>\n$list</ul>\n";
		}

		return self::get_wrapped_section( 'malware', $status, $description, $details );
	}

	protected static function get_malware_details( $results ) {
		$status = 'clean';
		$description = __( 'Malware', 'better-wp-security' );
		$details = false;

		$list = '';

		if ( ! empty( $results['MALWARE']['WARN'] ) ) {
			$status = 'warn';

			foreach ( (array) $results['MALWARE']['WARN'] as $entry ) {
				list( $message, $url ) = explode( ':', $entry[0], 2 );
				$message = trim( $message );
				$url = trim( $url );

				$message_lowercase = strtolower( $message );

				if ( 'security warning in the url' === $message_lowercase ) {
					$message = __( 'Security warning in the URL', 'better-wp-security' );
				} else if ( 'malware found on url' === $message_lowercase ) {
					$message = __( 'Malware found on URL', 'better-wp-security' );
				}

				if ( '&amp;' === substr( $url, 0, 5 ) ) {
					$payload = html_entity_decode( $url );
					$url = get_site_url();
				}

				$message .= "<br>\n";
				$message .= sprintf( __( 'Infected URL: <a href="%1$s" target="_blank" rel="noopener noreferrer">%2$s</a>', 'better-wp-security' ), esc_url( $url ), esc_html( $url ) );

				$parts = explode( "\n", $entry[1], 2 );

				if ( isset( $parts[1] ) ) {
					if ( preg_match( '/(.+)\. Details: (.+)/', $parts[0], $match ) ) {
						$type = $match[1];
						$documentation_url = $match[2];

						if ( '*Known Spam detected' === $type ) {
							$type = __( '*Known Spam detected', 'better-wp-security' );
						}

						$message .= "<br>\n";
						$message .= sprintf( __( 'Type: %1$s', 'better-wp-security' ), $type );
						$message .= "<br>\n";
						$message .= sprintf( __( 'Documentation: <a href="%1$s" target="_blank" rel="noopener noreferrer">%2$s</a>', 'better-wp-security' ), esc_url( $documentation_url ), esc_html( $documentation_url ) );
					}

					$payload_raw = trim( $parts[1] );
					$payload_raw = html_entity_decode( $payload_raw );

					if ( preg_match( '/<div id=\'HiddenDiv\'>(.+)<\/div>/', $payload_raw, $match ) ) {
						$payload = trim( $match[1] );
					}
				}

				if ( ! empty( $payload ) ) {
					$message .= "<br>\n";
					$message .= sprintf( __( 'Payload:<pre>%s</pre>', 'better-wp-security' ), esc_html( $payload ) );
				}

				$list .= sprintf( '<li class="itsec-malware-scan-warn"><span>%s</span></li>', $message ) . "\n";
			}
		}

		if ( ! empty( $list ) ) {
			$details = "<ul>\n$list</ul>\n";
		}

		return self::get_wrapped_section( 'malware', $status, $description, $details );
	}

	protected static function get_system_error_details( $results ) {
		if ( empty( $results['SYSTEM']['ERROR'] ) ) {
			return '';
		}

		$status = 'error';
		$description = __( 'The scan failed to properly scan the site.', 'better-wp-security' );

		$details = "<ul>\n";

		foreach ( $results['SYSTEM']['ERROR'] as $entry ) {
			$details .= sprintf( '<li class="itsec-malware-scan-error"><span>%s</span></li>', $entry ) . "\n";
		}

		$details .= "</ul>\n";

		return self::get_wrapped_section( 'system-error', $status, $description, $details );
	}

	protected static function get_wrapped_section( $type, $status, $description, $details = false ) {
		$html = "<div class='itsec-malware-scan-results-section itsec-malware-scan-results-$type-section'>\n";

		if ( 'clean' === $status ) {
			$status_text = __( 'Clean', 'better-wp-security' );
		} else if ( 'warn' === $status ) {
			$status_text = __( 'Warning', 'better-wp-security' );
		} else if ( 'error' === $status ) {
			$status_text = __( 'Error', 'better-wp-security' );
		}

		$status = "<span class='itsec-malware-scan-$status'>$status_text</span>";

		if ( false === $details ) {
			/* translators: Scan result listing. %1$s is the status. %2$s is the description. */
			$html .= "<p>" . sprintf( _x( '%1$s %2$s', 'scan status, scan description', 'better-wp-security' ), $status, $description ) . "</p>\n";
		} else {
			$details_link = "<a href='#' class='itsec-malware-scan-toggle-details'>" . __( 'Show Details', 'better-wp-security' ) . "</a>\n";

			/* translators: Scan result listing. %1$s is the status. %2$s is the description. %3$s is the details link. */
			$html .= "<p>" . sprintf( _x( '%1$s %2$s %3$s', 'scan status, scan description, scan details link', 'better-wp-security' ), $status, $description, $details_link ) . "</p>\n";

			$html .= "<div class='itsec-malware-scan-details'>\n$details\n</div>\n";
		}

		$html .= "</div>\n";

		return $html;
	}
}
